﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using Autodesk.Revit.DB;
using Autodesk.Revit.UI;
using Autodesk.Revit.UI.Selection;

namespace ZGF.Revit
{
    public partial class ElevationViewsDialog : System.Windows.Forms.Form
    {
        // M E M B E R S
        Document m_doc;
        List<InteriorElevation> m_int_elevation_views = new List<InteriorElevation>();
        
        MySortableBindingList<InteriorElevation> m_int_elevation_filtered;

        List<InteriorElevation> tmp_views_not_editable = new List<InteriorElevation>();
        
        double m_north_angle_offset = 0;

        Selection theCurrentlySelectedElements = null;

        bool m_user_pressed_cancel = false;

        string m_cell_value_at_click = string.Empty;

        // C O N S T R U C T O R
        public ElevationViewsDialog(ExternalCommandData commandData)
        {
            InitializeComponent();
            
            this.Text = "Interior Elevation Views by Room";
            this.toolStripStatusLabelVersion.Text = "Version: " + ZGF.Revit.Properties.Settings.Default.AppVersion;
            

            m_doc = commandData.Application.ActiveUIDocument.Document;

            theCurrentlySelectedElements = commandData.Application.ActiveUIDocument.Selection;

            // If project has a rotated True North, this will adjust
            // the orientation portion of the view name.
            ProjectLocationSet pls = m_doc.ProjectLocations;
            foreach (ProjectLocation p in pls)
            {
                if (p.Name == "Internal")
                {
                    ProjectPosition pos = p.get_ProjectPosition(XYZ.Zero);
                    m_north_angle_offset = Math.Round(InteriorElevation.RadiansToDegrees(pos.Angle), 3);
                    break;
                }
            } 

            

            dataGridViewsAll.AutoGenerateColumns = false;

            
            checkBoxLevel.Checked = false;
            checkBoxLevel_CheckedChanged(new object(), new EventArgs()); // Need to fire eventhandler because just setting value as above doesn't trigger

            CheckboxViewType.Checked = false;
            CheckboxViewType_CheckedChanged(new object(), new EventArgs());

            // Workshared projects:
            colEditable.Visible = colNotes.Visible = false;
            colEditedBy.Visible = m_doc.IsWorkshared;

            colViewName.AutoSizeMode = colNewViewName.AutoSizeMode = colTitleOnSheet.AutoSizeMode = colLevel.AutoSizeMode = colEditedBy.AutoSizeMode = colNotes.AutoSizeMode =
                 DataGridViewAutoSizeColumnMode.DisplayedCells;

            // Collect views, organize by room and post to datagrid:
            InitializeViewList();

            if (m_int_elevation_views.Count > 0)
                this.toolStripStatusLabelViewCount.Text = (m_int_elevation_views.Count + tmp_views_not_editable.Count).ToString() + " Views";

            
        }




        // H E L P E R    M E T H O D S

        private bool RenameInteriorElevationViews()
        {
            buttonOK.Enabled = false; // Disable OK


            
            
            toolStripProgressBar1.Value = 0;           
            toolStripProgressBar1.Maximum = m_int_elevation_views.Count;
            toolStripStatusLabelProgress.Text = "Renaming views...";
            this.toolStripProgressBar1.Width = statusStrip1.Width - (toolStripStatusLabelVersion.Width + toolStripStatusLabelViewCount.Width + toolStripStatusLabelProgress.Width);

            
            Transaction tr = new Transaction(m_doc, "Rename Interior Elevations");
            tr.Start();
            
            foreach (InteriorElevation iev in m_int_elevation_views)
            {
                try
                {
                    if (iev.Editable & iev.AutoNumber) 
                        iev.RenameView();
                    else
                    {
                        // collect un-editable view for posting at end of process
                        tmp_views_not_editable.Add(iev);
                        if (!iev.EditedBy.Equals(m_doc.Application.Username, StringComparison.OrdinalIgnoreCase))
                        {
                            iev.Remarks = "Cannot modify until " + iev.EditedBy + " saves to central";
                            iev.NewViewName = "";
                        }
                    }
                }
                catch (Exception ex)
                {
                    iev.Remarks = ex.Message;
                    iev.NewViewName = "";
                    tmp_views_not_editable.Add(iev);
                    continue;
                }
                finally
                {
                    toolStripProgressBar1.Increment(1);
                }

                Application.DoEvents();

                if (m_user_pressed_cancel)
                {
                    DialogResult result = MessageBox.Show("Really want to cancel?", "Renaming Views...", MessageBoxButtons.YesNo);
                    if (result == DialogResult.No)
                        m_user_pressed_cancel = false;
                    else
                    {
                        tr.RollBack();
                        //tr.Dispose();
                        return true;
                    }
                }

               
            }

            toolStripProgressBar1.Value = 0;

            tr.Commit();
            //tr.Dispose(); // is this necessary?

            // Grid displays views not editable.
            dataGridViewsAll.DataSource = null;
            m_int_elevation_views = tmp_views_not_editable;
            
            dataGridViewsAll.DataSource = m_int_elevation_views;
            if (m_int_elevation_views.Count > 0)
            {
                toolStripStatusLabelProgress.Text = m_int_elevation_views.Count.ToString() + " views not renamed";
                colTitleOnSheet.ReadOnly = true;
                colNotes.Visible = true;
                colNewViewName.Visible = false;
                buttonOK.Text = "Close";
                buttonOK.Enabled = true;
                buttonCancel.Enabled = false;
                return false;
            }
            else
            {
                // If make it all the way to the end...
                toolStripStatusLabelProgress.Text = "";
                
                return true;
            }

            
            
        }

        

        private void InitializeViewList()
        {
            //
            //  TODO: ProgressBar
            //
            
            m_int_elevation_views.Clear();
            dataGridViewsAll.DataSource = null;

            // Collect Elevation views
            FilteredElementCollector collector = new FilteredElementCollector(m_doc).OfClass(typeof(ViewSection));
            FilteredElementIterator iter = collector.GetElementIterator();
            while (iter.MoveNext())
            {
                ViewSection current = iter.Current as ViewSection;
                if (current == null) continue;

                ViewFamilyType vft = m_doc.GetElement(current.GetTypeId()) as ViewFamilyType;

                if (null == vft) continue;

                string typeName = vft.Name.ToLower();

                if (typeName.Contains("int") & typeName.Contains("elev")) // <--WARNING: This is an arbitrary rule !
                {                    
                    InteriorElevation ie = new InteriorElevation(current, m_north_angle_offset, m_doc);

                    if (ie.HasRoom)
                    {
                        // Check for duplicates
                        int dupeNumber = 0;
                        while (m_int_elevation_views.Count(Element => Element.NewViewName == ie.NewViewName) > 0)
                        {
                            dupeNumber++;
                            if (dupeNumber > 1)
                            {
                                string[] nameParts = ie.NewViewName.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                                // Exchange last item
                                nameParts[nameParts.Length - 1] = dupeNumber.ToString();
                                ie.NewViewName = string.Join(" ", nameParts);
                            }
                            else
                                ie.NewViewName += " " + dupeNumber.ToString();
                        }
                        // Add room to list    
                        m_int_elevation_views.Add(ie);

                    }
                    else
                    {
                        ie.Remarks = "Not inside room";
                        tmp_views_not_editable.Add(ie);
                    }
                }
            }

            // Populate grid
            if (m_int_elevation_views.Count == 0)
            {
                TaskDialog td = new TaskDialog("No Rooms");
                td.MainContent = "Could not find any Interior Elevation view tags inside Rooms...";
                td.Show();
                this.DialogResult = DialogResult.Cancel;
                this.Close();
            }

                        

            dataGridViewsAll.DataSource = new MySortableBindingList<InteriorElevation>(m_int_elevation_views);

            buttonOK.Enabled = dataGridViewsAll.Rows.Count > 0;

        }

        //-------------------------------------------------------------------------
        // F O R M   E V E N T   H A N D L E R S
        //-------------------------------------------------------------------------

        private void buttonCancel_Click(object sender, EventArgs e)
        {
            
            this.DialogResult = DialogResult.Cancel;
            m_user_pressed_cancel = true;
            this.Close();
        }

        private void buttonOK_Click(object sender, EventArgs e)
        {
            if (buttonOK.Text == "Close")
            {
                this.DialogResult = DialogResult.OK;
                this.Close();
            }
            else
            {
                
                Application.DoEvents();
                
                if (RenameInteriorElevationViews()) this.DialogResult = DialogResult.OK;

                textBoxSearchTerms.Clear();
                m_user_pressed_cancel = false; // made it to the end w/o user hitting cancel
            }
            
        }

       

        private void buttonSearchTerms_Click(object sender, EventArgs e)
        {
            textBoxSearchTerms.Clear();
        }

        private void textBoxSearchTerms_TextChanged(object sender, EventArgs e)
        {
            string words = this.textBoxSearchTerms.Text;
            dataGridViewsAll.DataSource = null;            
            m_int_elevation_filtered = new ZGF.Revit.MySortableBindingList<InteriorElevation>(m_int_elevation_views.FindAll(item => item.IsMatch(words)));
            this.dataGridViewsAll.DataSource = m_int_elevation_filtered;            
        }

        private void CheckboxViewType_CheckedChanged(object sender, EventArgs e)
        {
            colViewType.Visible = CheckboxViewType.Checked;
        }

        private void checkBoxLevel_CheckedChanged(object sender, EventArgs e)
        {
            colLevel.Visible = checkBoxLevel.Checked;
        }

        private void ElevationViewsDialog_SizeChanged(object sender, EventArgs e)
        {
            this.toolStripProgressBar1.Width = statusStrip1.Width - (toolStripStatusLabelVersion.Width + toolStripStatusLabelViewCount.Width + toolStripStatusLabelProgress.Width);

        }

        private void contextMenuStrip1_Opening(object sender, CancelEventArgs e)
        {
            // Check to see if any cells selected in 'Title on Sheet' column:
            int selectedCount = GetSelectedCellsIn_TitleOnSheet_Column().Count;
            if (null == dataGridViewsAll.SelectedCells | !buttonCancel.Enabled)
                e.Cancel = true;
            else
                e.Cancel = selectedCount > 0 ? false : true;
            
            
        }

        private void dataGridViewsAll_CellMouseClick(object sender, DataGridViewCellMouseEventArgs e)
        {
            
            
            if (e.Button == MouseButtons.Right)
                if (e.ColumnIndex == 2)
                {
                    m_cell_value_at_click = dataGridViewsAll[e.ColumnIndex, e.RowIndex].Value as string;
                    if (GetSelectedCellsIn_TitleOnSheet_Column().Count == 0) dataGridViewsAll[e.ColumnIndex, e.RowIndex].Selected = true;
                    contextMenuStrip1.Show(Cursor.Position);
                }
        }

        private void toolStripMenuItemSetValue_Click(object sender, EventArgs e)
        {  
            
            List<DataGridViewCell> cells = GetSelectedCellsIn_TitleOnSheet_Column();
            foreach (DataGridViewCell cell in cells)
            {
                if (cell.ColumnIndex == 2)
                    cell.Value = m_cell_value_at_click;
            }
        }

        private void toolStripMenuItemClear_Click(object sender, EventArgs e)
        {
            List<DataGridViewCell> cells = GetSelectedCellsIn_TitleOnSheet_Column();
            foreach (DataGridViewCell cell in cells)
                cell.Value = string.Empty;
        }

        private void toolStripMenuItemTitleEqualName_Click(object sender, EventArgs e)
        {
            List<DataGridViewCell> cells = GetSelectedCellsIn_TitleOnSheet_Column();
            foreach (DataGridViewCell cell in cells)
                cell.Value = dataGridViewsAll[1, cell.RowIndex].Value;
        }

      

        private List<DataGridViewCell> GetSelectedCellsIn_TitleOnSheet_Column()
        {
            List<DataGridViewCell> subCells = new List<DataGridViewCell>();

            DataGridViewSelectedCellCollection cells = dataGridViewsAll.SelectedCells;
            foreach (DataGridViewCell cell in cells)
            {
                if (cell.ColumnIndex == 2)
                {
                    subCells.Add(cell);
                }
            }

            return subCells;
        }


        
    }

    /// <summary>
    /// Wrapper class to manage Interior Elevation view properties
    /// </summary>
    class InteriorElevation : IComparable
    {
        // M E M B E R S
        ViewSection m_interiorElevView;

        bool m_UserNumberName = true;
        string m_editedBy = string.Empty;
        string m_currentUser;

        string m_titleOnSheet = string.Empty;

        // C O N S T R U C T O R
        public InteriorElevation(ViewSection intElevView, double NorthAngleOffset, Document currentDoc)
        {
            m_interiorElevView = intElevView;
            m_currentUser = currentDoc.Application.Username;

            XYZ pt = new XYZ(
                             m_interiorElevView.CropBox.Max.X - ((m_interiorElevView.CropBox.Max.X - m_interiorElevView.CropBox.Min.X) / 2),
                             m_interiorElevView.CropBox.Min.Y + ((m_interiorElevView.CropBox.Max.Y - m_interiorElevView.CropBox.Min.Y) / 3), // 1/3 of crop ht should get inside room
                             m_interiorElevView.CropBox.Max.Z);

            // Get pt's translation to project's coordinate system:            
            pt = m_interiorElevView.CropBox.Transform.OfPoint(pt);

            // Get the room...
            //  Note: The current phase of the elevation is the most likely room.            
            ElementId id = m_interiorElevView.get_Parameter(BuiltInParameter.VIEW_PHASE).AsElementId();
            Phase phase = currentDoc.GetElement(id) as Phase;
            Autodesk.Revit.DB.Architecture.Room rm = currentDoc.GetRoomAtPoint(pt, phase);
            if (null != rm)
            {
                HasRoom = true;
                AutoNumber = true; //TODO: 
                
                RoomNumber = rm.Number;
                RoomName = rm.get_Parameter("Name").AsString();
                ViewOrientation = GetCardinalDirectionIndicator(ConvertViewDirectionToAngle(m_interiorElevView.ViewDirection), NorthAngleOffset);
                UseNumberName = false;
                TitleOnSheet = m_interiorElevView.get_Parameter("Title on Sheet").AsString();

                ViewType = currentDoc.GetElement(m_interiorElevView.GetTypeId()).Name;
                LevelName = rm.Level.Name;
                
                Remarks = "";

                if (currentDoc.IsWorkshared)
                {
                    m_editedBy = m_interiorElevView.get_Parameter("Edited by").AsString();
                    Editable = m_editedBy.Equals(m_currentUser, StringComparison.OrdinalIgnoreCase) | m_editedBy.Equals(string.Empty);
                }
                else
                    Editable = true;
            }
            else
            {
                HasRoom = false;
            }
        }

        public bool HasRoom { get; set; }
        
        public bool AutoNumber { get; set; } // TODO: Make sure this Parameter exists
        public string ViewName { get { return m_interiorElevView.Name; } } //set; }

        public string TitleOnSheet
        {
            get { return m_titleOnSheet; }
            set { m_titleOnSheet = value != null ? value : string.Empty; }
        }

        public string RoomNumber { get; set; }
        public string RoomName { get; set; }
        public string ViewOrientation { get; set; }
        public string NewViewName { get; set; }

        public string LevelName { get; set; }
        public string ViewType { get; set; }

        public string Remarks { get; set; }
        public string EditedBy { get { return m_editedBy; }} 

        public bool Editable { get; set; }
        public bool UseNumberName { 
            get {return m_UserNumberName;}
            set
            {
                m_UserNumberName = value;
                if (m_UserNumberName)
                {
                    NewViewName =
                    RoomNumber + " " +
                    RoomName + " " +
                    ViewOrientation;
                }
                else
                {
                    NewViewName =
                    RoomName + " " +
                    RoomNumber + " " +
                    ViewOrientation;
                }
            }
        }

        public string SearchTerm { get { return ViewName + " " + NewViewName + " " + TitleOnSheet + " " + LevelName + " " + ViewType + " " + EditedBy; } }

        //-------------------------------------------------------------------------
        // P U B L I C   M E T H O D S
        //-------------------------------------------------------------------------

        // Rename room, but need to provide a transaction at calling method
        public bool RenameView()
        {
           
            try
            {
                if (m_interiorElevView.Name != NewViewName)  // What if same named view exists?
                    m_interiorElevView.Name = NewViewName;
             
            }
            catch
            {
                Remarks = "View name may already exist";
            }
           

            Parameter tmp_titleOnSheet = m_interiorElevView.get_Parameter("Title on Sheet");
            if (tmp_titleOnSheet.AsString() != TitleOnSheet)
                tmp_titleOnSheet.Set(TitleOnSheet);
            return true;

        }

        
        //-------------------------------------------------------------------------
        // P R I V A T E   H E L P E R   M E T H O D S
        //-------------------------------------------------------------------------

        public static double RadiansToDegrees(double radians)
        {
            return radians * (180 / Math.PI);
        }

        private double DegreesToRadians(double degrees)
        {
            return degrees * (Math.PI / 180);
        }

        private double NormalXY_ToRadians(double X, double Y)
        {
            return Math.Atan2(Y, X);
        }

        private double ConvertViewDirectionToAngle(XYZ ViewDirectionVector) // What about Project North direction?
        {
            // Get angle in radians:
            double rads = Math.Atan2(ViewDirectionVector.Y, ViewDirectionVector.X);
            // vector is toward elevation flag, so flip to get direction view is facing:
            if (rads < Math.PI)
                rads += Math.PI;
            else
                rads -= Math.PI;

            // Convert radians to degrees:
            return rads * (180 / Math.PI);

        }

        private string GetCardinalDirectionIndicator(double angle, double northangleOffset)
        {
            angle = Math.Round(angle, 4) + northangleOffset;
            string returnVal = "";
            if ((angle >= 0) && (angle < 45)) returnVal = "- EAST";
            if ((angle >= 45) && (angle < 135)) returnVal = "- NORTH";
            if ((angle >= 135) && (angle < 225)) returnVal = "- WEST";
            if ((angle >= 225) && (angle < 315)) returnVal = "- SOUTH";
            if ((angle >= 315) && (angle <= 360)) returnVal = "- EAST";

            return returnVal;

        }

        // requirement of iComparable
        public bool IsMatch(string searchWords)
        {
            char[] delimiters = new char[] { ' ', System.Convert.ToChar(160) };
            string stringToMatch = this.SearchTerm.ToLower();
            string literal = searchWords.ToLower();
            string[] words = literal.Split(delimiters, StringSplitOptions.RemoveEmptyEntries);

            int matches = 0;

            foreach (string s in words)
            {
                if (literal == string.Empty) continue;

                if ((stringToMatch.Contains(literal)) || (stringToMatch.Contains(s)))
                    matches++;
                else
                    matches--;
            }

            return (matches == words.Length);
        }

        // requirement of iComparable
        public int CompareTo(object obj)
        {

            InteriorElevation item = (InteriorElevation)obj;
            string s1 = item.ViewName + item.ViewName;
            string s2 = this.ViewName + this.ViewName;

            return string.Compare(s2, s1);
        }
    }

}
